#!/bin/bash
# SecureLink Pi Gateway Setup Script
# Configures a Raspberry Pi (3/Zero 2W) as an IKEv2 VPN gateway appliance.
#
# Network topology:
#   eth0 (WAN) → Starlink router (DHCP client)
#   wlan0 (LAN) → WiFi AP for downstream devices (192.168.4.0/24)
#   IPsec tunnel → VPN server (vpn.securelinkvpn.io)
#
# Usage:
#   sudo ./setup.sh [--claim-dir /path/to/claim/certs]

set -euo pipefail

SECURELINK_DIR="/etc/securelink"
AGENT_DIR="/opt/securelink"
CLAIM_DIR="${1:-}"
PI_SERIAL=$(cat /proc/cpuinfo | grep Serial | awk '{print $3}' | sed 's/^0*//')

echo "============================================"
echo "SecureLink Pi Gateway Setup"
echo "Pi Serial: ${PI_SERIAL}"
echo "============================================"

# Must run as root
if [ "$(id -u)" -ne 0 ]; then
    echo "ERROR: This script must be run as root (sudo)"
    exit 1
fi

# ── 1. System packages ──────────────────────────────────────────────

echo "[1/8] Installing system packages..."
apt-get update -qq
apt-get install -y -qq \
    strongswan \
    strongswan-pki \
    libcharon-extra-plugins \
    hostapd \
    dnsmasq \
    iptables-persistent \
    python3-pip \
    python3-venv \
    jq \
    curl

# ── 2. Directory structure ──────────────────────────────────────────

echo "[2/8] Creating directory structure..."
mkdir -p "${SECURELINK_DIR}/claim"
mkdir -p "${SECURELINK_DIR}/device"
mkdir -p "${AGENT_DIR}"
mkdir -p /var/lib/securelink
mkdir -p /var/log/securelink

# Copy claim certificates if provided
if [ -n "${CLAIM_DIR}" ] && [ -d "${CLAIM_DIR}" ]; then
    echo "  Copying claim certificates from ${CLAIM_DIR}..."
    cp "${CLAIM_DIR}/claim-cert.pem" "${SECURELINK_DIR}/claim/"
    cp "${CLAIM_DIR}/claim-key.pem" "${SECURELINK_DIR}/claim/"
    cp "${CLAIM_DIR}/aws-root-ca.pem" "${SECURELINK_DIR}/claim/"
    chmod 600 "${SECURELINK_DIR}/claim/claim-key.pem"
fi

# ── 3. WiFi Access Point (hostapd) ─────────────────────────────────

echo "[3/8] Configuring WiFi access point..."

# Stop services while configuring
systemctl stop hostapd 2>/dev/null || true
systemctl stop dnsmasq 2>/dev/null || true

# Unblock WiFi
rfkill unblock wlan 2>/dev/null || true

# Install hostapd config
SSID="SecureLink-${PI_SERIAL: -6}"
cp /dev/stdin /etc/hostapd/hostapd.conf <<HOSTAPD_EOF
# SecureLink Gateway WiFi AP
interface=wlan0
driver=nl80211
ssid=${SSID}
hw_mode=g
channel=7
wmm_enabled=0
macaddr_acl=0
auth_algs=1
ignore_broadcast_ssid=0
wpa=2
wpa_passphrase=SecureLink2026
wpa_key_mgmt=WPA-PSK
wpa_pairwise=CCMP
rsn_pairwise=CCMP
HOSTAPD_EOF

# Point hostapd to our config
sed -i 's|^#DAEMON_CONF=.*|DAEMON_CONF="/etc/hostapd/hostapd.conf"|' /etc/default/hostapd 2>/dev/null || true

# ── 4. DHCP Server (dnsmasq) ───────────────────────────────────────

echo "[4/8] Configuring DHCP server..."

# Back up original dnsmasq config
[ -f /etc/dnsmasq.conf.orig ] || cp /etc/dnsmasq.conf /etc/dnsmasq.conf.orig

cat > /etc/dnsmasq.conf <<DNSMASQ_EOF
# SecureLink Gateway DHCP/DNS
interface=wlan0
bind-interfaces
dhcp-range=192.168.4.10,192.168.4.200,255.255.255.0,24h
dhcp-option=option:router,192.168.4.1
dhcp-option=option:dns-server,192.168.4.1
server=8.8.8.8
server=8.8.4.4
log-queries
log-dhcp
DNSMASQ_EOF

# ── 5. Network configuration ───────────────────────────────────────

echo "[5/8] Configuring networking..."

# Static IP for wlan0 (AP interface)
cat >> /etc/dhcpcd.conf <<DHCPCD_EOF

# SecureLink: Static IP for WiFi AP interface
interface wlan0
    static ip_address=192.168.4.1/24
    nohook wpa_supplicant
DHCPCD_EOF

# Enable IP forwarding
sed -i 's/^#net.ipv4.ip_forward=1/net.ipv4.ip_forward=1/' /etc/sysctl.conf
echo "net.ipv4.ip_forward=1" >> /etc/sysctl.d/99-securelink.conf

# iptables NAT rules
# Default: masquerade LAN traffic through eth0 (before VPN is up)
# Once VPN is up, the agent will add rules to route through the tunnel
iptables -t nat -A POSTROUTING -o eth0 -j MASQUERADE
iptables -A FORWARD -i wlan0 -o eth0 -j ACCEPT
iptables -A FORWARD -i eth0 -o wlan0 -m state --state RELATED,ESTABLISHED -j ACCEPT

# Save iptables rules
netfilter-persistent save

# ── 6. strongSwan configuration ────────────────────────────────────

echo "[6/8] Configuring strongSwan..."

# Base ipsec.conf — the provisioning agent will write the connection config
cat > /etc/ipsec.conf <<IPSEC_EOF
# SecureLink Pi Gateway - strongSwan Configuration
# Connection config is managed by the securelink-agent

config setup
    charondebug="ike 2, knl 2, cfg 2, net 2"
    uniqueids=replace

include /etc/ipsec.d/*.conf
IPSEC_EOF

cat > /etc/ipsec.secrets <<SECRETS_EOF
# SecureLink Pi Gateway - strongSwan Secrets
# Device certificate key is referenced by the provisioning agent

include /etc/ipsec.d/*.secrets
SECRETS_EOF

# ── 7. Provisioning agent ──────────────────────────────────────────

echo "[7/8] Installing provisioning agent..."

# Create Python virtual environment
python3 -m venv "${AGENT_DIR}/venv"
"${AGENT_DIR}/venv/bin/pip" install --quiet \
    awsiotsdk \
    requests \
    cryptography

# Copy agent files
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
if [ -d "${SCRIPT_DIR}/securelink-agent" ]; then
    cp -r "${SCRIPT_DIR}/securelink-agent/"* "${AGENT_DIR}/"
fi

# Write device serial for the agent to read
echo "${PI_SERIAL}" > "${SECURELINK_DIR}/serial"

# Install systemd service
cat > /etc/systemd/system/securelink-agent.service <<SERVICE_EOF
[Unit]
Description=SecureLink Provisioning Agent
After=network-online.target
Wants=network-online.target
StartLimitIntervalSec=300
StartLimitBurst=5

[Service]
Type=simple
ExecStart=${AGENT_DIR}/venv/bin/python ${AGENT_DIR}/agent.py
WorkingDirectory=${AGENT_DIR}
Restart=on-failure
RestartSec=30
StandardOutput=journal
StandardError=journal
SyslogIdentifier=securelink-agent
Environment=SECURELINK_DIR=${SECURELINK_DIR}

[Install]
WantedBy=multi-user.target
SERVICE_EOF

systemctl daemon-reload
systemctl enable securelink-agent

# ── 8. Enable services ─────────────────────────────────────────────

echo "[8/8] Enabling services..."
systemctl unmask hostapd
systemctl enable hostapd
systemctl enable dnsmasq

# Start hostapd and dnsmasq now if possible
systemctl start hostapd || echo "  hostapd will start on next boot"
systemctl start dnsmasq || echo "  dnsmasq will start on next boot"

echo ""
echo "============================================"
echo "Setup complete!"
echo ""
echo "  WiFi SSID: ${SSID}"
echo "  WiFi Pass: SecureLink2026"
echo "  LAN IP:    192.168.4.1"
echo "  Serial:    ${PI_SERIAL}"
echo ""
echo "Next steps:"
echo "  1. Place claim certs in ${SECURELINK_DIR}/claim/"
echo "  2. Register this serial with: register-pi-device.py --serial ${PI_SERIAL}"
echo "  3. Reboot: sudo reboot"
echo "============================================"
