"""
VPN Manager for SecureLink Pi Gateway.

Handles:
- Calling the activation API to get VPN certificates
- Writing strongSwan configuration
- Managing the IPsec tunnel lifecycle
- Updating iptables for gateway routing through the tunnel
"""

import base64
import json
import logging
import subprocess
import time
from pathlib import Path

import requests
from cryptography.hazmat.primitives.serialization import pkcs12, Encoding, PrivateFormat, NoEncryption
from cryptography.x509 import load_pem_x509_certificate

from config import (
    API_URL, DEVICE_DIR,
    VPN_CA_CERT, VPN_DEVICE_CERT, VPN_DEVICE_KEY,
    IPSEC_CONF, IPSEC_SECRETS,
)

log = logging.getLogger('securelink.vpn')


class VpnManager:
    """Manages strongSwan IKEv2 VPN connection."""

    def __init__(self):
        self.vpn_server = 'vpn.securelinkvpn.io'
        self.device_id = None
        self.assigned_ip = None

    def is_configured(self) -> bool:
        """Check if VPN certificates and config are in place."""
        return (VPN_CA_CERT.exists()
                and VPN_DEVICE_CERT.exists()
                and VPN_DEVICE_KEY.exists()
                and IPSEC_CONF.exists())

    def activate(self, activation_code: str) -> dict:
        """
        Call the SecureLink activation API to get VPN certificates.
        Returns the activation response.
        """
        log.info('Calling activation API...')

        url = f'{API_URL}/activate'
        payload = {'activation_code': activation_code}

        resp = requests.post(url, json=payload, timeout=30)
        resp.raise_for_status()
        data = resp.json()

        self.device_id = data.get('device_id')
        self.assigned_ip = data.get('assigned_ip')
        self.vpn_server = data.get('vpn_server', self.vpn_server)

        log.info(f'Activated device {self.device_id}, static IP: {self.assigned_ip}')

        # Extract and store certificates
        self._store_certificates(data)

        # Write strongSwan config
        self._write_ipsec_config()

        # Save VPN config to disk
        vpn_config = {
            'device_id': self.device_id,
            'assigned_ip': self.assigned_ip,
            'vpn_server': self.vpn_server,
            'activated_at': time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime()),
        }
        vpn_config_path = DEVICE_DIR / 'vpn-config.json'
        vpn_config_path.write_text(json.dumps(vpn_config, indent=2))

        return vpn_config

    def _store_certificates(self, activation_data: dict):
        """Extract certs from activation response and write to disk."""
        # CA certificate (PEM)
        ca_cert_pem = activation_data.get('ca_certificate', '')
        if ca_cert_pem:
            VPN_CA_CERT.write_text(ca_cert_pem)
            log.info(f'Wrote CA certificate to {VPN_CA_CERT}')

            # Also install CA cert for strongSwan
            ca_ipsec_path = Path('/etc/ipsec.d/cacerts/securelink-ca.pem')
            ca_ipsec_path.parent.mkdir(parents=True, exist_ok=True)
            ca_ipsec_path.write_text(ca_cert_pem)

        # Device certificate (PKCS#12 base64-encoded)
        p12_b64 = activation_data.get('device_certificate', '')
        p12_password = activation_data.get('certificate_password', '')

        if p12_b64:
            p12_bytes = base64.b64decode(p12_b64)
            password_bytes = p12_password.encode() if p12_password else None

            # Parse PKCS#12
            private_key, certificate, chain = pkcs12.load_key_and_certificates(
                p12_bytes, password_bytes
            )

            # Write device certificate (PEM)
            cert_pem = certificate.public_bytes(Encoding.PEM)
            VPN_DEVICE_CERT.write_text(cert_pem.decode())
            log.info(f'Wrote device certificate to {VPN_DEVICE_CERT}')

            # Also install for strongSwan
            cert_ipsec_path = Path('/etc/ipsec.d/certs/device-cert.pem')
            cert_ipsec_path.parent.mkdir(parents=True, exist_ok=True)
            cert_ipsec_path.write_text(cert_pem.decode())

            # Write private key (PEM, no encryption - protected by file perms)
            key_pem = private_key.private_bytes(
                Encoding.PEM, PrivateFormat.PKCS8, NoEncryption()
            )
            VPN_DEVICE_KEY.write_text(key_pem.decode())
            VPN_DEVICE_KEY.chmod(0o600)
            log.info(f'Wrote device private key to {VPN_DEVICE_KEY}')

            # Also install for strongSwan
            key_ipsec_path = Path('/etc/ipsec.d/private/device-key.pem')
            key_ipsec_path.parent.mkdir(parents=True, exist_ok=True)
            key_ipsec_path.write_text(key_pem.decode())
            key_ipsec_path.chmod(0o600)

    def _write_ipsec_config(self):
        """Write strongSwan connection config matching the Libreswan server."""
        config = f"""# SecureLink VPN Connection
# Auto-generated by securelink-agent
# Device: {self.device_id}
# Static IP: {self.assigned_ip}

conn securelink
    type=tunnel
    keyexchange=ikev2
    left=%defaultroute
    leftcert=device-cert.pem
    leftid="{self._get_cert_subject()}"
    right={self.vpn_server}
    rightid=%any
    rightsubnet=0.0.0.0/0
    ike=aes256-sha256-modp2048!
    esp=aes256gcm16-modp2048!
    auto=start
    dpdaction=restart
    dpddelay=300s
    dpdtimeout=60s
    closeaction=restart
    keyingtries=%forever
    reauth=no
    fragmentation=yes
"""
        IPSEC_CONF.parent.mkdir(parents=True, exist_ok=True)
        IPSEC_CONF.write_text(config)
        log.info(f'Wrote strongSwan config to {IPSEC_CONF}')

        # Write secrets file referencing the private key
        secrets = "# SecureLink VPN Secrets\n: RSA device-key.pem\n"
        IPSEC_SECRETS.write_text(secrets)
        log.info(f'Wrote strongSwan secrets to {IPSEC_SECRETS}')

    def _get_cert_subject(self) -> str:
        """Read the subject DN from the device certificate."""
        if VPN_DEVICE_CERT.exists():
            cert_pem = VPN_DEVICE_CERT.read_bytes()
            cert = load_pem_x509_certificate(cert_pem)
            # Format as strongSwan expects: "CN=device-id, O=SecureLink"
            return cert.subject.rfc4514_string()
        # Fallback
        return f'CN={self.device_id}, O=SecureLink' if self.device_id else ''

    def connect(self):
        """Start the strongSwan IPsec tunnel."""
        log.info('Starting IPsec tunnel...')
        self._run('ipsec restart', check=False)
        time.sleep(3)

        # Wait for tunnel to come up
        for attempt in range(10):
            if self.is_connected():
                log.info('IPsec tunnel established')
                self._update_gateway_routing()
                return True
            log.info(f'Waiting for tunnel (attempt {attempt + 1}/10)...')
            time.sleep(5)

        log.error('Failed to establish IPsec tunnel')
        return False

    def reconnect(self):
        """Restart the IPsec connection."""
        log.info('Reconnecting IPsec tunnel...')
        self._run('ipsec down securelink', check=False)
        time.sleep(2)
        self._run('ipsec up securelink', check=False)
        time.sleep(5)
        if self.is_connected():
            self._update_gateway_routing()
            return True
        return False

    def is_connected(self) -> bool:
        """Check if the IPsec tunnel is up."""
        result = self._run('ipsec status securelink', check=False)
        return 'ESTABLISHED' in result

    def get_status(self) -> dict:
        """Get VPN connection status details."""
        status = {
            'vpn_connected': self.is_connected(),
            'device_id': self.device_id,
            'assigned_ip': self.assigned_ip,
            'vpn_server': self.vpn_server,
        }

        if status['vpn_connected']:
            # Get tunnel details from ipsec statusall
            details = self._run('ipsec statusall securelink', check=False)
            status['tunnel_details'] = details[:500]  # Truncate for MQTT

        return status

    def _update_gateway_routing(self):
        """Update iptables to route LAN traffic through the VPN tunnel."""
        # Remove the old eth0 masquerade for LAN traffic
        self._run(
            'iptables -t nat -D POSTROUTING -o eth0 -j MASQUERADE',
            check=False
        )
        # Add masquerade through the IPsec tunnel
        # strongSwan uses virtual IP or policy-based routing; traffic matching
        # the tunnel policy is automatically routed through IPsec.
        # We re-add eth0 masquerade for non-tunneled traffic (DNS etc.)
        self._run(
            'iptables -t nat -A POSTROUTING -o eth0 -j MASQUERADE',
            check=False
        )
        log.info('Gateway routing updated for VPN tunnel')

    def _run(self, cmd: str, check: bool = True) -> str:
        """Run a shell command."""
        result = subprocess.run(
            cmd, shell=True, capture_output=True, text=True
        )
        if check and result.returncode != 0:
            log.error(f'Command failed: {cmd}\n{result.stderr}')
        return result.stdout.strip()

    def load_saved_config(self):
        """Load previously saved VPN config from disk."""
        vpn_config_path = DEVICE_DIR / 'vpn-config.json'
        if vpn_config_path.exists():
            config = json.loads(vpn_config_path.read_text())
            self.device_id = config.get('device_id')
            self.assigned_ip = config.get('assigned_ip')
            self.vpn_server = config.get('vpn_server', self.vpn_server)
            return True
        return False
