"""
Status Reporter for SecureLink Pi Gateway.

Publishes device telemetry to AWS IoT Core via MQTT every 60 seconds.
"""

import json
import logging
import os
import subprocess
import time
from pathlib import Path

from awscrt import mqtt
from awsiot import mqtt_connection_builder

from config import (
    DEVICE_CERT, DEVICE_KEY, AWS_ROOT_CA, DEVICE_CONFIG,
    IOT_ENDPOINT, STATUS_INTERVAL,
)

log = logging.getLogger('securelink.status')


class StatusReporter:
    """Reports device status/telemetry to IoT Core via MQTT."""

    def __init__(self, vpn_manager):
        self.vpn = vpn_manager
        self.connection = None
        self.thing_name = None
        self._boot_time = time.time()

    def start(self):
        """Connect to IoT Core and begin status reporting loop."""
        if not DEVICE_CONFIG.exists():
            log.warning('No device config - skipping status reporting')
            return

        config = json.loads(DEVICE_CONFIG.read_text())
        self.thing_name = config.get('thing_name')

        if not IOT_ENDPOINT or not self.thing_name:
            log.warning('IoT endpoint or thing name not configured')
            return

        log.info(f'Starting status reporter as {self.thing_name}')

        self.connection = mqtt_connection_builder.mtls_from_path(
            endpoint=IOT_ENDPOINT,
            cert_filepath=str(DEVICE_CERT),
            pri_key_filepath=str(DEVICE_KEY),
            ca_filepath=str(AWS_ROOT_CA),
            client_id=self.thing_name,
            clean_session=True,
            keep_alive_secs=30,
        )

        self.connection.connect().result(timeout=30)
        log.info('Connected to IoT Core for status reporting')

    def report_once(self):
        """Publish a single status report."""
        if not self.connection or not self.thing_name:
            return

        status = self._collect_status()
        topic = f'securelink/{self.thing_name}/status'

        self.connection.publish(
            topic=topic,
            payload=json.dumps(status),
            qos=mqtt.QoS.AT_LEAST_ONCE,
        ).result(timeout=10)

        log.debug(f'Published status to {topic}')

    def run_loop(self):
        """Run the status reporting loop (blocking)."""
        self.start()

        while True:
            try:
                self.report_once()
            except Exception:
                log.exception('Status report failed')
                # Try to reconnect
                try:
                    self.start()
                except Exception:
                    log.exception('Reconnect failed')

            time.sleep(STATUS_INTERVAL)

    def stop(self):
        """Disconnect from IoT Core."""
        if self.connection:
            try:
                self.connection.disconnect().result(timeout=10)
            except Exception:
                pass

    def _collect_status(self) -> dict:
        """Collect device status and telemetry."""
        vpn_status = self.vpn.get_status()

        return {
            'thing_name': self.thing_name,
            'timestamp': time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime()),
            'status': {
                'vpn_connected': vpn_status.get('vpn_connected', False),
                'static_ip': vpn_status.get('assigned_ip', ''),
                'uptime_seconds': int(time.time() - self._boot_time),
                'lan_clients': self._count_lan_clients(),
                'cpu_percent': self._get_cpu_usage(),
                'memory_percent': self._get_memory_usage(),
                'temperature_c': self._get_cpu_temperature(),
            },
            'network': {
                'wan_ip': self._get_wan_ip(),
                'rx_bytes': self._get_interface_bytes('eth0', 'rx'),
                'tx_bytes': self._get_interface_bytes('eth0', 'tx'),
            },
        }

    def _count_lan_clients(self) -> int:
        """Count connected LAN clients from dnsmasq leases."""
        lease_file = Path('/var/lib/misc/dnsmasq.leases')
        if lease_file.exists():
            return len(lease_file.read_text().strip().splitlines())
        return 0

    def _get_cpu_usage(self) -> float:
        """Get CPU usage percentage."""
        try:
            load1 = float(Path('/proc/loadavg').read_text().split()[0])
            # Normalize to percentage (Pi 3 has 4 cores)
            return round(min(load1 / 4.0 * 100, 100), 1)
        except Exception:
            return 0.0

    def _get_memory_usage(self) -> float:
        """Get memory usage percentage."""
        try:
            meminfo = Path('/proc/meminfo').read_text()
            total = available = 0
            for line in meminfo.splitlines():
                if line.startswith('MemTotal:'):
                    total = int(line.split()[1])
                elif line.startswith('MemAvailable:'):
                    available = int(line.split()[1])
            if total > 0:
                return round((1 - available / total) * 100, 1)
        except Exception:
            pass
        return 0.0

    def _get_cpu_temperature(self) -> float:
        """Get CPU temperature in Celsius."""
        try:
            temp_path = Path('/sys/class/thermal/thermal_zone0/temp')
            if temp_path.exists():
                return round(int(temp_path.read_text().strip()) / 1000, 1)
        except Exception:
            pass
        return 0.0

    def _get_wan_ip(self) -> str:
        """Get WAN interface IP."""
        try:
            result = subprocess.run(
                "ip -4 addr show eth0 | grep 'inet ' | awk '{print $2}' | cut -d/ -f1",
                shell=True, capture_output=True, text=True
            )
            return result.stdout.strip()
        except Exception:
            return ''

    def _get_interface_bytes(self, interface: str, direction: str) -> int:
        """Get bytes received or transmitted on an interface."""
        try:
            stat = 'rx_bytes' if direction == 'rx' else 'tx_bytes'
            path = Path(f'/sys/class/net/{interface}/statistics/{stat}')
            if path.exists():
                return int(path.read_text().strip())
        except Exception:
            pass
        return 0
