#!/usr/bin/env python3
"""
SecureLink Pi Gateway Agent

Main entry point for the provisioning and VPN management daemon.
Runs as a systemd service on Raspberry Pi.

Boot sequence:
  1. Read Pi serial number
  2. If no IoT device cert → fleet provisioning (claim cert exchange)
  3. If no VPN config → get config via MQTT → call activation API
  4. Start strongSwan VPN tunnel
  5. Enter status reporting loop
"""

import json
import logging
import os
import subprocess
import sys
import time
from pathlib import Path

# Add agent directory to path
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

from config import (
    SECURELINK_DIR, DEVICE_DIR, DEVICE_CONFIG, STATE_FILE,
    SERIAL_FILE, STATUS_INTERVAL,
)
from iot_provisioner import IoTProvisioner
from vpn_manager import VpnManager
from status_reporter import StatusReporter

logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s [%(levelname)s] %(name)s: %(message)s',
    handlers=[logging.StreamHandler(sys.stdout)],
)
log = logging.getLogger('securelink.agent')


def get_serial() -> str:
    """Read the Pi serial number."""
    if SERIAL_FILE.exists():
        return SERIAL_FILE.read_text().strip()
    # Read from /proc/cpuinfo
    try:
        with open('/proc/cpuinfo') as f:
            for line in f:
                if line.startswith('Serial'):
                    return line.split(':')[1].strip().lstrip('0')
    except Exception:
        pass
    raise RuntimeError('Cannot determine Pi serial number')


def get_mac() -> str:
    """Read the primary MAC address."""
    try:
        mac_path = Path('/sys/class/net/eth0/address')
        if mac_path.exists():
            return mac_path.read_text().strip().upper()
        # Fall back to wlan0
        mac_path = Path('/sys/class/net/wlan0/address')
        if mac_path.exists():
            return mac_path.read_text().strip().upper()
    except Exception:
        pass
    return '00:00:00:00:00:00'


def wait_for_network(timeout: int = 120):
    """Wait for network connectivity."""
    log.info('Waiting for network...')
    start = time.time()
    while time.time() - start < timeout:
        try:
            result = subprocess.run(
                ['ping', '-c', '1', '-W', '3', '8.8.8.8'],
                capture_output=True, timeout=10,
            )
            if result.returncode == 0:
                log.info('Network is up')
                return True
        except Exception:
            pass
        time.sleep(5)
    log.error('Network timeout')
    return False


def main():
    log.info('SecureLink Pi Gateway Agent starting')

    serial = get_serial()
    mac = get_mac()
    log.info(f'Serial: {serial}, MAC: {mac}')

    # Wait for network before proceeding
    if not wait_for_network():
        log.error('No network connectivity - exiting')
        sys.exit(1)

    vpn = VpnManager()
    provisioner = IoTProvisioner(serial, mac)

    # ── Phase 1: IoT Fleet Provisioning ──────────────────────────

    if not provisioner.is_provisioned():
        log.info('Phase 1: IoT fleet provisioning')
        try:
            device_config = provisioner.provision()
            log.info(f'Provisioned as: {device_config.get("thing_name")}')
        except Exception:
            log.exception('Fleet provisioning failed')
            log.info('Retrying in 60 seconds...')
            time.sleep(60)
            sys.exit(1)  # systemd will restart us
    else:
        log.info('Phase 1: Already provisioned')

    # ── Phase 2: VPN Activation ──────────────────────────────────

    if not vpn.is_configured():
        log.info('Phase 2: VPN activation')
        try:
            # Get VPN config (activation code) from IoT Core
            vpn_config = provisioner.get_config_via_mqtt()
            activation_code = vpn_config.get('activation_code')

            if not activation_code:
                log.error('No activation code in config')
                time.sleep(60)
                sys.exit(1)

            # Call activation API
            result = vpn.activate(activation_code)
            log.info(f'VPN activated, static IP: {result.get("assigned_ip")}')

        except Exception:
            log.exception('VPN activation failed')
            time.sleep(60)
            sys.exit(1)
    else:
        log.info('Phase 2: VPN already configured')
        vpn.load_saved_config()

    # ── Phase 3: Connect VPN ─────────────────────────────────────

    log.info('Phase 3: Connecting VPN tunnel')
    if not vpn.connect():
        log.error('VPN connection failed - will retry on restart')
        time.sleep(30)
        sys.exit(1)

    log.info(f'VPN connected! Static IP: {vpn.assigned_ip}')

    # ── Phase 4: Status Reporting ────────────────────────────────

    log.info('Phase 4: Starting status reporting')
    reporter = StatusReporter(vpn)

    try:
        reporter.start()
    except Exception:
        log.exception('Status reporter failed to start (continuing without)')

    # Main loop: monitor VPN and report status
    while True:
        try:
            # Check VPN health
            if not vpn.is_connected():
                log.warning('VPN tunnel down - reconnecting')
                vpn.reconnect()

            # Report status
            try:
                reporter.report_once()
            except Exception:
                log.exception('Status report failed')

        except Exception:
            log.exception('Main loop error')

        time.sleep(STATUS_INTERVAL)


if __name__ == '__main__':
    main()
